define([
    'underscore',
    'backbone',
    'marionette',
    'App',
    'modules/new-appointment-request/views/common/type-of-care/layout-view',
    'modules/new-appointment-request/views/common/city-state/layout-view',
    'modules/new-appointment-request/views/common/facility/layout-view',
    'modules/new-appointment-request/views/common/method/layout-view',
    'modules/new-appointment-request/views/direct/layout-view',
    'modules/new-appointment-request/views/requests/layout-view',
    'modules/new-appointment-request/resources/common/pac-team/collection',
    'text!modules/new-appointment-request/views/common/scheduling-steps/templates/template.html',
    'GoogleAnalyticsProvider',
], function(
    _,
    Backbone,
    Mn,
    app,
    TypeOfCare,
    CityState,
    Facility,
    SchedulingMethod,
    AppointmentsView,
    RequestsView,
    PactCollection,
    template,
    GoogleAnalyticsProvider
) {
    'use strict';


    /**
     * @class SectionLayoutView
     * @typedef {Backbone.View<Backbone.Model>}
     */
    var View = Mn.View.extend({
        template: _.template(template),

        regions: {
            'typeOfCare': '.scheduling-type-of-care',
            'city-state': '.scheduling-city-state',
            'facility': '.scheduling-facility',
            'scheduling-method': '.scheduling-method',
            'details': '.scheduling-details',
            'requests': '.scheduling-requests',
        },

        modelEvents: {
            'change:typeOfCare': 'showCityState',
            'change:cityState': 'showFacility',
            'change:facility': 'showMethod',
            'change:scheduling-method': 'onMethodChange',
        },

        childViewEvents: {
            'store:clinics': 'storeClinics',
        },

        storeClinics: function(childView, clincs) {
            this.clinicsCollection = clincs;
        },

        onRender: function() {
            this.showTypeOfCare();
        },

        showTypeOfCare: function() {
            var region = this.getRegion(View.TYPE_OF_CARE);
            var typeOfCare = new TypeOfCare({
                model: this.model,
                googleAnalyticsProvider: GoogleAnalyticsProvider,
            });

            this._resetAfter(View.TYPE_OF_CARE);
            region.show(typeOfCare);
        },

        /**
         * @param {Backbone.Model} facilityModel
         * @param {Backbone.Model} facility
         * @return {boolean} whether or not a PAC team link is needed for a
         * Primary Care request at a certain location
         */
        isPacTeamLinkNeeded: function(facilityModel, facility) {
            var unfilteredClinics = facility.get('unfilteredClinics') || new Backbone.Collection();
            var facilityInstitutionCode = facilityModel.get('institutionCode');
            var institutionCodes;
            var containsInstitutionCode;

            institutionCodes = unfilteredClinics.map(function(clinic) {
                return clinic.get('institutionCode');
            });
            
            containsInstitutionCode = institutionCodes.indexOf(facilityInstitutionCode) !== -1;

            if (_.isUndefined(app.disabledFeatures.get('PACT_LOCATION_FEATURE_SET'))) {
                return (containsInstitutionCode && this.model.isPrimaryCare()) || this.model.isExpressCare();
            }

            return this.model.isPrimaryCare() || this.model.isExpressCare();
        },

        _showCityState: function() {
            var region = this.getRegion(View.CITY_STATE);
            region.show(new CityState({
                model: this.model,
            }));
        },

        showCityState: function() {
            this._resetAfter(View.CITY_STATE);

            if (this.model.get('typeOfCare')) {
                this._showCityState();
            }
        },

        _showFacility: function() {
            var region = this.getRegion(View.FACILITY);
            var typeRegion = this.getRegion(View.TYPE_OF_CARE);
            var typeView = typeRegion.currentView;
            var typeOfCareName = typeView.getCareName();
            this.model.set('pacTeam', new PactCollection());

            region.show(new Facility({
                model: this.model,
                typeOfCareName: typeOfCareName,
                googleAnalyticsProvider: GoogleAnalyticsProvider,
            }));
        },

        showFacility: function() {
            this._resetAfter(View.FACILITY);

            if (this.model.get('typeOfCare')) { // TODO switch this to city-state
                this._showFacility();
            }
        },

        showMethod: function(model, facilityId) {
            var region = this.getRegion(View.METHOD);

            var facilityModel = this.model.facility();
            var isPacTeamNeeded = this.isPacTeamLinkNeeded(facilityModel, model);
            var pactCollection;

            this._resetAfter(View.METHOD);
            region.show(new SchedulingMethod({
                model: this.model,
                typeOfCareName: this.model.getCareName(),
                isRequestSupported: facilityModel.isRequestSupported(),
                isDirectSchedulingSupported: facilityModel.isDirectSchedulingSupported(),
                institutionCode: this.model.getInstitutionCode(),
                isPacNeeded: isPacTeamNeeded,
            }));
            if (isPacTeamNeeded) {
                pactCollection = this.model.pacTeam();
                pactCollection.fetch({}, this.model.getInstitutionCode());
            }
        },

        onMethodChange: function(model, schedulingMethod) {
            var details = this.getRegion('details');
            var requests = this.getRegion('requests');

            details.empty();
            requests.empty();
            if (schedulingMethod === 'direct') {
                this.showAppointmentDetails();
            } else if (schedulingMethod === 'clerk') {
                this.showRequestDetails();
            }
        },

        showAppointmentDetails: function() {
            var region = this.getRegion('details');
            var options = {
                model: this.model,
            };

            if (!_.isEmpty(this.clinicsCollection)) {
                options.clinicsCollection = this.clinicsCollection;
            }
            region.show(new AppointmentsView(options));
        },

        showRequestDetails: function() {
            var region = this.getRegion('requests');

            region.show(new RequestsView({model: this.model}));
        },

        _resetRegion: function(name) {
            var region = this.getRegion(name);
            var view = region.currentView;

            this.model.unset(name, {silent: true});

            if (view) {
                region.reset();
            }
        },

        _resetAfter: function(regionName) {
            var isPassed = false;

            this.triggerMethod('hide:email');
            _.each(View.ORDER, function(name) {
                if (isPassed || regionName === name) {
                    isPassed = true;
                    this._resetRegion(name);
                    if (name === View.METHOD) {
                        this.clinicsCollection = null;
                    }
                }
            }.bind(this));
        },
    });


    View.TYPE_OF_CARE = 'typeOfCare';
    View.METHOD = 'scheduling-method';
    View.FACILITY = 'facility';
    View.CITY_STATE = 'city-state';
    View.DETAILS = 'details';
    View.REQUEST = 'requests';

    View.ORDER = [
        View.TYPE_OF_CARE,
        View.CITY_STATE,
        View.FACILITY,
        View.METHOD,
        View.DETAILS,
        View.REQUEST,
    ];


    return View;
});
